<?php

namespace Fivable\Scaffolding\Controllers;

use App\Http\Controllers\Controller;
use App\User;
use Auth;
use Carbon\Carbon;
use DB;
use Fivable\Scaffolding\Requests\LoginRequest;
use Fivable\Scaffolding\Mail\ForgotPassword;
use Fivable\Scaffolding\Models\JwtToken;
use Fivable\Scaffolding\Traits\HandlesTokens;
use Fivable\Scaffolding\Traits\HasUser;
use Hash;
use Illuminate\Contracts\View\Factory;
use Illuminate\Foundation\Auth\Access\AuthorizesRequests;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\Routing\Redirector;
use Illuminate\View\View;
use Log;
use Mail;

class ApiController extends Controller
{
    use HasUser, HandlesTokens, AuthorizesRequests;

    /**
     * Get the login form for !headless
     *
     * @param Request $request
     * @return Factory|RedirectResponse|Redirector|View
     */
    public function getLogin(Request $request) {
        if (! is_null($this->user())) {
            return redirect('/' . config('scaffolding.routes.home'));
        }
        return view(config('scaffolding.get_views_from') . '::login');
    }

    /**
     * Logs the user in based on their email/username, password, and current
     * Bearer token
     *
     * Note: The LoginRequest will transform username into email if possible
     *
     * @param  LoginRequest $request
     * @return mixed
     */
    public function login(LoginRequest $request)
    {
        if (Auth::attempt(['email' => $request->email, 'password' => $request->password])) {

            $user = Auth::user();
            Log::info("Login by user with id: " . $user->id);

            $bearer_token = $request->bearerToken();
            $expiry = Carbon::now()->addDays(1);

            if ($bearer_token == 'null' || $bearer_token == 'undefined' || $bearer_token == null) {
                $token_object = $this->createJwtObject($user->id, $expiry);
                $jwt_token = $token_object->jwt_token;

                if (config('scaffolding.headless-api')) {
                    return response()->json([
                        'success' => 'Logged in',
                        'jwt_token' => $jwt_token,
                    ]);
                } else {
                    $request->session()->flash('flash_message', 'Logged in');
                    return redirect('/' . config('scaffolding.routes.home'));
                }

            } else if ($bearer_token != null) {

                $token_object = JwtToken::where('jwt_token', $bearer_token)->first();

                if ($token_object == null || $token_object == 'undefined' || $token_object == 'null') {
                    $token_object = $this->createJwtObject($user->id, $expiry);
                }

                if (Carbon::now()->gt($token_object->expiry)) {
                    $token_object->jwt_token = $this->createToken($user->id, $expiry);
                    $token_object->expiry = $expiry;
                    $token_object->save();
                }

                $jwt_token = $token_object->jwt_token;

                if (config('scaffolding.headless-api')) {
                    return response()->json([
                        'success' => 'Logged in',
                        'jwt_token' => $jwt_token,
                    ]);
                } else {
                    $request->session()->flash('flash_message', 'Logged in');
                    return redirect('/' . config('scaffolding.routes.home'));
                }
            }

        } else {
            if (config('scaffolding.headless-api')) {
                return response()->json([
                    'error' => 'The email/username and/or password entered is invalid',
                ], 401);
            } else {
                $errors = ['The email/username and/or password entered is invalid'];
                return view(config('scaffolding.get_views_from') . '::login', compact('errors'));
            }
        }
    }

    /**
     * Logout the current User, delete their JwtToken, and flush it from session
     *
     * @param Request $request
     * @return JsonResponse|RedirectResponse|Redirector
     */
    public function logout(Request $request)
    {
        // Ensure InsertToken middleware is cleared on logout
        if ($request->session()->has('jwt_token')) {
            $request->session()->forget('jwt_token');
            $request->session()->flush();
        }

        JwtToken::where('user_id', $this->user()->id)->delete();
        Auth::logout();

        if (config('scaffolding.headless-api')) {
            return response()->json(['success' => 'User was logged out']);
        } else {
            $request->session()->flash('flash_message', 'Logged out');
            return redirect('/' . config('scaffolding.routes.login'));
        }
    }

    /**
     * For !headless, get the Forgot Password form
     *
     * Note: You can optionally just hit the forgotPassword route from the login
     * form, which has all the relevant inputs
     *
     * @param Request $request
     * @return Factory|View
     */
    public function getForgotPassword(Request $request)
    {
        return view(config('scaffolding.get_views_from') . '::forgot-password');
    }

    /**
     * Send the Forgot Password email, including Reset Password link
     *
     * @param Request $request
     * @return JsonResponse|RedirectResponse|Redirector
     */
    public function forgotPassword(Request $request)
    {
        $user = User::where('email', $request->email)
                    ->orWhere('username', $request->username)
                    ->first();
        if (is_null($user)) {
            if (config('scaffolding.headless-api')) {
                return response()->json(['error' => 'User not found'], 404);
            } else {
                $request->session()->flash('flash_message', 'User record was not found');
                return redirect('/' . config('scaffolding.routes.login'));
            }
        }
        $hash = Hash::make($user->id . '|' . $user->email);
        $email = urlencode($user->email);
        DB::table('password_resets')->insert([
            'email' => $user->email, // non-encoded
            'token' => $hash,
            'created_at' => Carbon::now()
        ]);
        Mail::to($user->email)->send(new ForgotPassword(base64_encode($hash), $email));

        if (config('scaffolding.headless-api')) {
            return response()->json([
                'success' => 'A forgot password email has been sent to ' . $user->email,
            ]);
        } else {
            $request->session()->flash('flash_message', 'An email has been sent to ' . $user->email);
            return redirect('/' . config('scaffolding.routes.login'));
        }
    }

    /**
     * Get the Reset Password form (not for headless API)
     *
     * Note: Forgot Password can use email or username, but the generated link
     * that leads to this endpoint ALWAYS uses the User's email
     *
     * @param Request $request
     * @param string  $hash
     * @param string  $email
     * @return mixed
     */
    public function getResetPassword(Request $request, string $hash, string $email)
    {
        $user = User::where('email',urldecode($email))->first();
        $password_reset = DB::table('password_resets')->where('email', urldecode($email))
            ->where('token', base64_decode($hash))->where('created_at', '>', Carbon::now()->addMinutes(-15))
            ->first();

        if (is_null($user)) {
            $errors = ['User not found'];
            return view(config('scaffolding.get_views_from') . '::forgot-password', compact('errors'));
        } else if (is_null($password_reset)) {
            $errors = ['Password Reset not found or expired'];
            return view(config('scaffolding.get_views_from') . '::forgot-password', compact('errors'));
        }

        $token_object = JwtToken::where('user_id', $user->id)->first();
        if (!is_null($token_object)) {
            $jwt_token = $token_object->jwt_token;
        } else {
            $expiry = Carbon::now()->addDays(1);
            $token_object = $this->createJwtObject($user->id, $expiry);
            $jwt_token = $token_object->jwt_token;
        }

        if(Hash::check($user->id . '|' . $user->email,base64_decode($hash))) {
            // $password_reset->delete();
            return view(config('scaffolding.get_views_from') . '::reset-password', compact('jwt_token'));
        } else {
            $errors = ['Invalid Password Reset link'];
            return view(config('scaffolding.get_views_from') . '::forgot-password', compact('errors'));
        }
    }

    /**
     * Allows headless front-ends to verify a reset password link to allow
     * access to their own Reset Password form
     *
     * @param Request $request
     * @return JsonResponse
     */
    public function verifyResetPassword(Request $request): JsonResponse
    {
        $hash = $request->hash;
        $emailOrUsername = urldecode($request->url_encoded_email_or_username);

        $user = User::where('email', $emailOrUsername)
                    ->orWhere('username', $emailOrUsername)
                    ->first();
        if (is_null($user)) {
            return response()->json([
                'error' => 'User not found',
            ], 404);
        }

        $passwordReset = DB::table('password_resets')->where('email', $user->email)
            ->where('token', base64_decode($hash))
            ->where('created_at', '>', Carbon::now()->addMinutes(-15))
            ->first();
        if (is_null($passwordReset)) {
            return response()->json([
                'error' => 'Password Reset not found or expired',
            ], 404);
        }

        $tokenObject = JwtToken::where('user_id', $user->id)->first();
        if (! is_null($tokenObject)) {
            $jwtToken = $tokenObject->jwt_token;
        } else {
            $expiry = Carbon::now()->addDays(1);
            $tokenObject = $this->createJwtObject($user->id, $expiry);
            $jwtToken = $tokenObject->jwt_token;
        }

        if(Hash::check($user->id . '|' . $user->email, base64_decode($hash))) {
            // TODO: Consider the password reset after use, as below
            // $passwordReset->delete();
            return response()->json([
                'success' => 'Password Reset is legitimate',
                'jwt_token' => $jwtToken,
            ]);
        } else {
            return response()->json([
                'error' => 'Invalid Password Reset link',
            ], 400);
        }
    }

    /**
     * Actually reset the User's password!
     *
     * @param Request $request
     * @return mixed (View if !headless, JsonResponse if headless)
     */
    public function resetPassword(Request $request)
    {
        $jwt_token = JwtToken::where('jwt_token', $request->jwt_token)->first();
        if (is_null($jwt_token)) {
            if (config('scaffolding.headless-api')) {
                return response()->json(['error' => 'Token not found'], 404);
            } else {
                return view(
                    config('scaffolding.get_views_from') . '::reset-password',
                    ['errors' => ['Token not found']]
                );
            }
        }

        $user = User::find($jwt_token->user_id);
        if (is_null($user)) {
            if (config('scaffolding.headless-api')) {
                return response()->json(['error' => 'User not found'], 404);
            } else {
                return view(
                    config('scaffolding.get_views_from') . '::reset-password',
                    ['errors' => ['User not found']]
                );
            }
        }

        if ($request->new_password === $request->again_password) {

            $user->password = Hash::make($request->new_password);
            $user->save();

            if (config('scaffolding.headless-api')) {
                return response()->json(['success' => 'Password was reset']);
            } else {
                $request->session()->flash('flash_message', 'Password Reset Successful');
                return redirect('/' . config('scaffolding.routes.login'));
            }

        } else {
            if (config('scaffolding.headless-api')) {
                return response()->json(['error' => 'New passwords did not match'], 400);
            } else {
                return view(config('scaffolding.get_views_from') . '::reset-password', ['errors' => ['New passwords did not match']]);
            }
        }
    }

    /**
     * Get the default Home view
     *
     * @param Request $request
     * @return Factory|View
     */
    public function defaultHome(Request $request) {
        return view(config('scaffolding.get_views_from') . '::default-home');
    }

    /**
     * Retrieves the currently authenticated User
     *
     * @param  Request  $request
     * @return JsonResponse
     */
    public function whoami(Request $request): JsonResponse
    {
        $bearer_token = $request->bearerToken();

        if ($bearer_token == null || $bearer_token == 'undefined' || $bearer_token == 'null') {

            return response()->json([
                'error' => 'No bearer token in whoami', 'user' => null
            ], 400);
        } else {

            $jwt_token = JwtToken::where('jwt_token', $bearer_token)->first();
            $user = User::find($jwt_token->user_id);

            if (is_null($user)) {
                return response()->json([
                    'error' => 'Null User in whoami', 'user' => null
                ], 400);
            }

            return response()->json([
                'success' => 'Successfully found User',
                'user' => $user,
            ]);
        }
    }
}
