<?php

namespace Fivable\Scaffolding\Controllers;

use App\Http\Controllers\Controller;
use App\User;
use Auth;
use Carbon\Carbon;
use DB;
use Firebase\JWT\JWT;
use Fivable\Scaffolding\Mail\ForgotPassword;
use Fivable\Scaffolding\Models\JwtToken;
use Fivable\Scaffolding\Traits\HasUser;
use Hash;
use Illuminate\Encryption\Encrypter;
use Illuminate\Foundation\Auth\Access\AuthorizesRequests;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Str;
use Log;
use Mail;

class ApiController extends Controller
{

    use HasUser, AuthorizesRequests;

    public function resetPassword(Request $request)
    {
        $jwt_token = JwtToken::where('jwt_token', $request->jwt_token)->first();
        if (is_null($jwt_token)) {
            if (config('scaffolding.headless-api')) {
                return response()->json(['error' => 'Token not found'], 404);
            } else {
                return view(config('scaffolding.get_views_from') . '::reset-password', ['errors' => ['Token not found']]);
            }
        }

        $user = User::find($jwt_token->user_id);
        if (is_null($user)) {
            if (config('scaffolding.headless-api')) {
                return response()->json(['error' => 'User not found'], 404);
            } else {
                return view(config('scaffolding.get_views_from') . '::reset-password', ['errors' => ['User not found']]);
            }
        }

        if ($request->new_password === $request->again_password) {

            $user->password = Hash::make($request->new_password);
            $user->save();

            if (config('scaffolding.headless-api')) {
                return response()->json(['success' => 'Password was reset']);
            } else {
                $request->session()->flash('flash_message', 'Password Reset Successful');
                return redirect('/' . config('scaffolding.routes.login'));
            }

        } else {
            if (config('scaffolding.headless-api')) {
                return response()->json(['error' => 'New passwords did not match'], 400);
            } else {
                return view(config('scaffolding.get_views_from') . '::reset-password', ['errors' => ['New passwords did not match']]);
            }
        }
    }

    /**
     * Creates a new JwtToken Object
     * @param  {int}    $id     User ID
     * @param  {Carbon} $expiry Expiry date/time
     * @return {JwtToken}            JWT to be used for subsequent logins
     */
    private function _createJwtObject($id, $expiry)
    {
        DB::table('jwt_tokens')->where('user_id', $id)->delete();

        $new_jwt = new JwtToken;
        $new_jwt->jwt_token = $this->_createToken($id, $expiry);
        $new_jwt->user_id = $id;
        $new_jwt->expiry = $expiry;
        $new_jwt->save();

        return $new_jwt;
    }

    /**
     * Logs the user in based on their email, password, and current Bearer token
     * @param  Request $request           Login request
     * @return mixed
     */
    public function login(Request $request)
    {
        if (Auth::attempt(['email' => $request->email, 'password' => $request->password])) {

            $user = Auth::user();
            Log::info("Login by user with id: " . $user->id);

            $bearer_token = $request->bearerToken();
            $expiry = Carbon::now()->addDays(1);

            if ($bearer_token == 'null' || $bearer_token == 'undefined' || $bearer_token == null) {
                $token_object = $this->_createJwtObject($user->id, $expiry);
                $jwt_token = $token_object->jwt_token;

                if (config('scaffolding.headless-api')) {
                    return response()->json([
                        'success' => 'Logged in',
                        'jwt_token' => $jwt_token,
                    ]);
                } else {
                    $request->session()->flash('flash_message', 'Logged in');
                    return redirect('/' . config('scaffolding.routes.home'));
                }

            } else if ($bearer_token != null) {

                $token_object = JwtToken::where('jwt_token', $bearer_token)->first();

                if ($token_object == null || $token_object == 'undefined' || $token_object == 'null') {
                    $token_object = $this->_createJwtObject($user->id, $expiry);
                }

                if (Carbon::now()->gt($token_object->expiry)) {
                    $token_object->jwt_token = $this->_createToken($user->id, $expiry);
                    $token_object->expiry = $expiry;
                    $token_object->save();
                }

                $jwt_token = $token_object->jwt_token;

                if (config('scaffolding.headless-api')) {
                    return response()->json([
                        'success' => 'Logged in',
                        'jwt_token' => $jwt_token,
                    ]);
                } else {
                    $request->session()->flash('flash_message', 'Logged in');
                    return redirect('/' . config('scaffolding.routes.home'));
                }
            }

        } else {
            if (config('scaffolding.headless-api')) {
                return response()->json([
                    'error' => 'The email or password entered is invalid',
                ], 401);
            } else {
                $errors = ['The email or password entered is invalid'];
                return view(config('scaffolding.get_views_from') . '::login', compact('errors'));
            }
        }
    }

    /**
     * Logs the user in based on their email, password, and current Bearer token
     * @param  Request $request           Login request
     * @return {JSON Response}            JSON response with token if successfully logged in
     */
    public function ucLogin($user)
    {
        Auth::login($user, true);

        if (Auth::user() == null) {
            Log::error("User is null in login!");
        }

        $expiry = Carbon::now()->addDays(1);
        $new_jwt = $this->_createJwtObject(Auth::user()->id, $expiry);

        return $new_jwt->jwt_token;
    }

    public function logout(Request $request)
    {
        // Ensure InsertToken middleware is cleared on logout
        if ($request->session()->has('jwt_token')) {
            $request->session()->forget('jwt_token');
            $request->session()->flush();
        }

        JwtToken::where('user_id', $this->user()->id)->delete();
        Auth::logout();

        if (config('scaffolding.headless-api')) {
            return response()->json(['success' => 'User was logged out']);
        } else {
            $request->session()->flash('flash_message', 'Logged out');
            return redirect('/' . config('scaffolding.routes.login'));
        }
    }

    /**
     * Creates a JWT
     *
     * @param  int      $userId
     * @param  Carbon   $expiration
     * @return string
     */
    private function _createToken(int $userId, Carbon $expiration) : string
    {
        $key = config('app.key');

        if (Str::startsWith($key, 'base64:')) {
            $key = base64_decode(substr($key, 7));
        }

        $encrypter = new Encrypter($key, config('app.cipher'));

        $jwt = JWT::encode([
          'sub' => $userId,
          'expiry' => $expiration->getTimestamp(),
        ], $encrypter->getKey());

        return $jwt;
    }

    public function forgotPassword(Request $request)
    {
        $user = User::where('email', $request->email)->first();
        if (is_null($user)) {
            if (config('scaffolding.headless-api')) {
                return response()->json(['error' => 'User not found'], 404);
            } else {
                $request->session()->flash('flash_message', 'User record was not found');
                return redirect('/' . config('scaffolding.routes.login'));
            }
        }
        $hash = Hash::make($user->id . '|' . $user->email);
        $email = urlencode($user->email);
        DB::table('password_resets')->insert([
            'email' => $user->email, //non-encoded
            'token' => $hash,
            'created_at' => Carbon::now()
        ]);
        Mail::to($user->email)->send(new ForgotPassword(base64_encode($hash),$email));

        if (config('scaffolding.headless-api')) {
            return response()->json([
                'success' => 'A forgot password email has been sent to ' . $user->email,
            ]);
        } else {
            $request->session()->flash('flash_message', 'An email has been sent to ' . $user->email);
            return redirect('/' . config('scaffolding.routes.login'));
        }
    }

    public function getResetPassword(Request $request, $hash, $email)
    {
        $user = User::where('email',urldecode($email))->first();
        $password_reset = DB::table('password_resets')->where('email', urldecode($email))
            ->where('token', base64_decode($hash))->where('created_at', '>', Carbon::now()->addMinutes(-15))
            ->first();

        if (is_null($user)) {
            $errors = ['User not found'];
            return view(config('scaffolding.get_views_from') . '::forgot-password', compact('errors'));
        } else if (is_null($password_reset)) {
            $errors = ['Password Reset not found or expired'];
            return view(config('scaffolding.get_views_from') . '::forgot-password', compact('errors'));
        }

        $token_object = JwtToken::where('user_id', $user->id)->first();
        if (!is_null($token_object)) {
            $jwt_token = $token_object->jwt_token;
        } else {
            $expiry = Carbon::now()->addDays(1);
            $token_object = $this->_createJwtObject($user->id, $expiry);
            $jwt_token = $token_object->jwt_token;
        }

        if(Hash::check($user->id . '|' . $user->email,base64_decode($hash))) {
            // $password_reset->delete();
            return view(config('scaffolding.get_views_from') . '::reset-password', compact('jwt_token'));
        } else {
            $errors = ['Invalid Password Reset link'];
            return view(config('scaffolding.get_views_from') . '::forgot-password', compact('errors'));
        }
    }

    /**
     * Allows headless Scaffolding apps to verify that the reset password link they got is legitimate
     *
     * @param Request $request
     * @return mixed
     */
    public function verifyResetPassword(Request $request)
    {
        $hash = $request->hash;
        $email = urldecode($request->url_encoded_email);
        $user = User::where('email',$email)->first();
        $password_reset = DB::table('password_resets')->where('email', $email)
            ->where('token', base64_decode($hash))->where('created_at', '>', Carbon::now()->addMinutes(-15))
            ->first();

        if (is_null($user)) {
            return response()->json([
                'error' => 'User not found',
            ], 404);
        } else if (is_null($password_reset)) {
            return response()->json([
                'error' => 'Password Reset not found or expired',
            ], 404);
        }

        $token_object = JwtToken::where('user_id', $user->id)->first();
        if (!is_null($token_object)) {
            $jwt_token = $token_object->jwt_token;
        } else {
            $expiry = Carbon::now()->addDays(1);
            $token_object = $this->_createJwtObject($user->id, $expiry);
            $jwt_token = $token_object->jwt_token;
        }

        if(Hash::check($user->id . '|' . $user->email,base64_decode($hash))) {
            // $password_reset->delete();
            return response()->json([
                'success' => 'Password Reset is legitimate',
                'jwt_token' => $jwt_token
            ]);
        } else {
            return response()->json([
                'error' => 'Invalid Password Reset link',
            ], 400);
        }
    }

    public function defaultHome(Request $request) {
        return view(config('scaffolding.get_views_from') . '::default-home');
    }

    public function getLogin(Request $request) {
        if (!is_null($this->user())) {
            return redirect('/' . config('scaffolding.routes.home'));
        }
        return view(config('scaffolding.get_views_from') . '::login');
    }

    public function getForgotPassword(Request $request) {
        return view(config('scaffolding.get_views_from') . '::forgot-password');
    }

    /**
     * Retrieves the currently authenticated User
     *
     * @param  Request  $request
     * @return JsonResponse
     */
    public function whoami(Request $request) : JsonResponse
    {
        $bearer_token = $request->bearerToken();

        if ($bearer_token == null || $bearer_token == 'undefined' || $bearer_token == 'null') {

            return response()->json([
                'error' => 'No bearer token in whoami', 'user' => null
            ], 400);
        } else {

            $jwt_token = JwtToken::where('jwt_token', $bearer_token)->first();
            $user = User::find($jwt_token->user_id);

            if (is_null($user)) {
                return response()->json([
                    'error' => 'Null User in whoami', 'user' => null
                ], 400);
            }

            return response()->json([
                'success' => 'Successfully found User', 'user' => $user
            ]);
        }
    }
}
