<?php

namespace Fivable\FivableFiles\Models;

use Carbon\Carbon;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Http\Request;
use Illuminate\Http\UploadedFile;
use Illuminate\Support\Facades\Storage;

/**
 * @method static create(array $array)
 * @property string mime_type
 * @property string original_file_name
 * @property string url
 * @property int    file_size
 * @property int    height
 * @property int    width
 * @property string storage_name
 * @property string storage_path
 * @property string extension
 * @property Carbon  deleted_at
 */
class File extends Model
{
    use SoftDeletes;

    protected $fillable = [
        'original_file_name',
        'url',
        'mime_type',
        'file_size',
        'width',
        'height',
    ];

    /**
     * Determine the 'storage_path' attribute
     * (Get the full path of the S3 file associated with this File)
     *
     * @return string
     */
    public function getStoragePathAttribute(): string
    {
        return $this->mime_type . '/' . $this->storage_name;
    }

    /**
     * Determine the 'storage_name' attribute
     * (The name of the S3 file but not the full path to it)
     *
     * @return string
     */
    public function getStorageNameAttribute(): string
    {
        return $this->id . '_' . $this->original_file_name;
    }

    /**
     * Determine the 'extension' attribute
     *
     * @return string
     */
    public function getExtensionAttribute(): string
    {
        $slashPosition = strpos($this->mime_type, '/');
        return substr($this->mime_type, $slashPosition + 1);
    }

    public static function createFile(Request $request)
    {
        if (! $request->hasFile('file')) {
            return false;
        }

        /** @var UploadedFile $uploadedFile */
        $uploadedFile = $request->file('file');

        if (self::fileIsImage($uploadedFile)) {
            $dimensions = self::getImageDimensions($uploadedFile);
        }

        /** @var File $file */
        $file = File::create([
            'original_file_name' => $uploadedFile->getClientOriginalName(),
            'mime_type' => $uploadedFile->getMimeType(),
            'file_size' => $uploadedFile->getSize(),
            'width' => $dimensions['width'] ?? null,
            'height' => $dimensions['height'] ?? null,
            'url' => '',
        ]);

        return $file->putInStorage($uploadedFile)->fresh();
    }

    public function updateFile(Request $request)
    {
        if (! $request->hasFile('file')) {
            return false;
        }

        $updatedFile = $request->file('file');

        $this->deleteInStorage();
        $this->putInStorage($updatedFile);

        if (self::fileIsImage($updatedFile)) {
            $dimensions = self::getImageDimensions($updatedFile);
        }

        $this->update([
            'original_file_name' => $this->getMostCurrentFilename($updatedFile),
            'mime_type' => $this->getMostCurrentFileExtension($updatedFile),
            'file_size' => $updatedFile->getSize(),
            'width' => $dimensions['width'] ?? null,
            'height' => $dimensions['height'] ?? null,
        ]);

        return $this->fresh();
    }

    /**
     * Persist a File in the configured S3 bucket
     *
     * @param UploadedFile $file
     * @return File
     */
    public function putInStorage(UploadedFile $file): File
    {
        Storage::disk('s3')->putFileAs(
            $this->mime_type,
            $file,
            $this->storage_name,
            ['visibility' => 'public']
        );

        $this->update([
            'url' => Storage::disk('s3')->url($this->storage_path),
        ]);

        return $this;
    }

    /**
     * Remove this File from the configured S3 bucket
     */
    public function deleteInStorage()
    {
        Storage::disk('s3')->delete($this->storage_path);
    }

    /**
     * Determine whether or not a file is an image
     *
     * @param UploadedFile $file
     * @return bool
     */
    protected static function fileIsImage(UploadedFile $file): bool
    {
        return substr($file->getMimeType(), 0, 5) === 'image';
    }

    /**
     * If the file is an image, get the height and width
     *
     * @param UploadedFile $file
     * @return array
     */
    protected static function getImageDimensions(UploadedFile $file): array
    {
        $imageData = getimagesize($file);

        return [
            'width' => $imageData[0],
            'height' => $imageData[1],
        ];
    }

    /**
     * Gets the most current filename for this file--either
     * the one associated with this object, or the
     * one contained within the given file
     *
     * @param UploadedFile $file
     * @return string
     */
    protected function getMostCurrentFilename(UploadedFile $file): string
    {
        return $file ?
            $file->getClientOriginalName() :
            $this->original_file_name;
    }

    /**
     * Gets the most current extension for this file--either
     * the one associated with this object, or the
     * one contained within the given file
     *
     * @param UploadedFile $file
     * @return string
     */
    protected function getMostCurrentFileExtension(UploadedFile $file): string
    {
        return $file ?
            $file->getClientOriginalExtension() :
            $this->mime_type;
    }
}
