<?php

namespace Fivable\Buildable\Policies;

use App\User;
use Illuminate\Auth\Access\HandlesAuthorization;

class UserPolicy
{
    use HandlesAuthorization;

    /**
     * If User is Superadmin, allow all
     * @param $user
     * @return bool|NULL
     */
    public function before(User $user)
    {
        if ($user->isSuperadmin()) {
            return true;
        }
    }

    /**
     * Determine whether the User can see an index of Users
     * @param  User  $user
     * @return bool
     */
    public function viewIndex(User $user) : bool
    {
        return !($user->isEditor());
    }

    /**
     * Determine whether the User can see this User
     * @param  User  $user
     * @param  User  $otherUser
     * @return bool
     */
    public function view(User $user, User $otherUser) : bool
    {
        $sameUsers = ($user->id == $otherUser->id);

        return $sameUsers || ($user->isAdmin() && !$otherUser->isSuperAdmin());
    }

    /**
     * Determine whether the User can create Users
     * @param  User  $user
     * @param  int|NULL  $roleMask
     * @return bool
     */
    public function create(User $user, int $roleMask = NULL) : bool
    {
        $notCreatingSuperadmin = ($roleMask != 1);

        return $user->isAdmin() && $notCreatingSuperadmin;
    }

    /**
     * Determine whether the User can update this User
     * @param  User  $user
     * @param  User  $otherUser
     * @param  int   $newRoleMask
     * @return bool
     */
    public function update(User $user, User $otherUser, int $newRoleMask) : bool
    {
        $sameUsers = $user->id == $otherUser->id;

        // If role_mask has been altered, verify the User trying to update is not changing bits above their pay grade
        if ($newRoleMask != $otherUser->role_mask) {
            $bitsTaken = array_filter($otherUser->roleMaskArray(), function ($oldBit) use ($newRoleMask) {
                return !($oldBit & $newRoleMask);
            });
            $bitsGiven = array_filter(User::staticMaskArray($newRoleMask), function ($newBit) use ($otherUser) {
                return !($newBit & $otherUser->role_mask);
            });

            $illegalBitsTaken = array_filter($bitsTaken, function ($takenBit) use ($user) {
                return $takenBit > $user->highestRoleMaskBit();
            });
            $illegalBitsGiven = array_filter($bitsGiven, function ($givenBit) use ($user) {
                return $givenBit > $user->highestRoleMaskBit();
            });
        }

        return ($sameUsers || !$otherUser->isSuperadmin()) && empty($illegalBitsTaken) && empty($illegalBitsGiven);
    }

    /**
     * Determine whether the user can delete the buildable user.
     *
     * @param  User  $user
     * @param  User  $otherUser
     * @return bool
     */
    public function delete(User $user, User $otherUser) : bool
    {
        $sameUsers = $user->id == $otherUser->id;

        $userOutrankedBy = array_filter($otherUser->roleMaskArray(), function ($bit) use ($user) {
            return $bit > $user->highestRoleMaskBit();
        });

        return $sameUsers || empty($userOutrankedBy);
    }
}
